#ifndef _AMI_THREAD_H_
#define _AMI_THREAD_H_

#include "AmiMutex.h"
#include <pthread.h>
#include <errno.h>
#include <unistd.h>
#include <sys/time.h>

namespace AmiSample {

// =======================================================================
//  ThreadIndex
// =======================================================================
typedef unsigned int ThreadIndex;

// =======================================================================
//  Event
// =======================================================================
class Event {
  public:
	Event()
	: isSignalSet_(false) {
		pthread_mutexattr_init(&mutexAttr_);
		pthread_mutex_init(&mutex_, &mutexAttr_);
		pthread_condattr_init(&handleAttr_);
		pthread_cond_init(&handle_, &handleAttr_);
	}

	inline virtual ~Event() {
		pthread_condattr_destroy(&handleAttr_);
		pthread_cond_destroy(&handle_);
		pthread_mutexattr_destroy(&mutexAttr_);
		pthread_mutex_destroy(&mutex_);
	}

	inline bool wait() {
		pthread_mutex_lock(&mutex_);
		while (!isSignalSet_) {
			pthread_cond_wait(&handle_, &mutex_);
		}
		isSignalSet_ = false;
		pthread_mutex_unlock(&mutex_);
		return true;
	}

	inline bool wait(unsigned int timeout) {
		struct timeval timeBeforeWaiting, timeAfterWaiting;
		struct timespec absoluteWaitingTime;
		pthread_mutex_lock(&mutex_);
		while (!isSignalSet_) {
			gettimeofday(&timeBeforeWaiting, NULL);
			long tmp = timeBeforeWaiting.tv_usec + 1000 * timeout;
			absoluteWaitingTime.tv_sec = timeBeforeWaiting.tv_sec + tmp / 1000000;
			absoluteWaitingTime.tv_nsec = (tmp % 1000000) * 1000;
			pthread_cond_timedwait(&handle_, &mutex_, &absoluteWaitingTime);
			if (isSignalSet_) {
				break;
			}

			gettimeofday(&timeAfterWaiting, NULL);
			unsigned int elapsedTime = (timeAfterWaiting.tv_sec - timeBeforeWaiting.tv_sec) * 1000
			                 + (timeAfterWaiting.tv_usec - timeBeforeWaiting.tv_usec) / 1000;
			if (elapsedTime >= timeout) {
				isSignalSet_ = false;
				pthread_mutex_unlock(&mutex_);
				return false;
			}
			timeout -= elapsedTime;
		}
		isSignalSet_ = false;
		pthread_mutex_unlock(&mutex_);
		return true;
	}

	inline void eatup() {
		pthread_mutex_lock(&mutex_);
		isSignalSet_ = false;
		pthread_mutex_unlock(&mutex_);
	}

	inline void signal() {
		pthread_mutex_lock(&mutex_);
		isSignalSet_ = true;
		pthread_cond_signal(&handle_);
		pthread_mutex_unlock(&mutex_);
		Event::sleep(0);
	}

	static inline void sleep(int milliSecs) {
		usleep(milliSecs * 1000);
	}

  protected:
	bool isSignalSet_;
	pthread_mutex_t mutex_;
	pthread_mutexattr_t mutexAttr_;
	pthread_cond_t handle_;
	pthread_condattr_t handleAttr_;
};

// =======================================================================
//  Thread
// =======================================================================
class Thread {
  protected:
	static void* threadEntry(void* arg) {
		Thread* thread = (Thread*)arg;
		thread->threadStarted_.wait();
		thread->run();
		{
			MutexLock lock(thread->mutex_);
			thread->isAlive_ = false;
		}
		thread->threadTerminated_.signal();
		pthread_exit(0);
		return NULL;
	}

	virtual void run() = 0;

  public:
	static inline ThreadIndex getCurrentThreadId() {
		return (ThreadIndex)pthread_self();
	}

	static inline void sleep(int milliSecs) {
		usleep(milliSecs * 1000);
	}

  public:
	Thread()
	: threadHandle_(0)
	, isAlive_(false) {
		pthread_attr_init(&threadAttr_);
		pthread_attr_setstacksize(&threadAttr_, 1024 * 1024);
		pthread_attr_setdetachstate(&threadAttr_, PTHREAD_CREATE_DETACHED);
	}

	virtual ~Thread() {
		join();
		if (threadHandle_ != 0) {
			pthread_attr_destroy(&threadAttr_);
			threadHandle_ = 0;
		}
	}

	void start() {
		MutexLock lock(mutex_);
		if (threadHandle_ == 0) {
			int error = pthread_create(&threadHandle_, &threadAttr_, threadEntry, this);
			if (error != 0) {
				threadHandle_ = 0;
				return;
			}
			isAlive_ = true;
			threadStarted_.signal();
		}
	}

	void join() {
		if (isAlive()) {
			threadTerminated_.wait();
		}
	}

	void join(unsigned int timeout) {
		unsigned int elapsedTime = 0;
		while(elapsedTime < timeout) {
			if (!isAlive()) {
				return;
			}
			threadTerminated_.wait(100);
			elapsedTime += 100;
		}
	}

	void lock() {
		mutex_.lock();
	}

	void unlock() {
		mutex_.unlock();
	}

	inline ThreadIndex getThreadId() {
		return (ThreadIndex)threadHandle_;
	}

	bool isAlive() {
		MutexLock lock(mutex_);
		return isAlive_;
	}

  private:
	pthread_attr_t threadAttr_;
	pthread_t threadHandle_;
	Event threadStarted_;
	Event threadTerminated_;
	Mutex mutex_;
	bool isAlive_;
};

} // namespace AMI

#endif /* _AMI_THREAD_H_ */
